;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Variable and Breed declarations ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

breeds [ cases orders backorders customers ] ;; These are game pieces

globals [
  ;; variables related to the game board
  retail-inventory     ;; number of cases in inventory (for retailer)
  distrib-inventory    ;;                              (for distributor)     
  manufact-inventory   ;;                              (for manufacturer)
  retail-backorders    ;; number of cases demanded but not delivered (for R)
  distrib-backorders   ;;                                            (for D)
  manufact-backorders  ;;                                            (for M)
  number-of-customers  ;; randomized number of customers for each turn
  shipping-line1       ;; number of cases on top-most white bar on right side
  shipping-line2       ;; number of cases on second from top bar on right side
  shipping-line3       ;; number of cases on third from top bar on right side
  shipping-line4       ;; number of cases on fourth from top bar on right side
  shipping-line5       ;; number of cases on fifth from top bar on right side
  shipping-line6       ;; number of cases on bottom-most from top bar on right side
  order-line1          ;; number of orders on top-most white bar on left side
  order-line2          ;; number of orders on middle white bar on left side
  order-line3          ;; number of orders on bottom white bar in left side
  
  ;; variables used for "accounting" 
  costs   ;; total costs incurred by team during game
  turns   ;; number of times through the "go" loop
  
  ;; variables used to interact with participants
  retailer_ID           ;; hubnet-message-source of retailer
  distributor_ID        ;;                       of distributor
  manufacturer_ID       ;;                       of manufacturer
  retailer-slider-value        ;; most recent value read in from retailer's slider
  distributor-slider-value     ;;                           from distributor's slider
  manufacturer-slider-value    ;;                           from manufcaturer's slider
]


;;;;;;;;;;;;;;;;;;;;;;;;
;;; Setup Procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;;

to startup
  hubnet-set-client-interface "COMPUTER" [ "clients/Beer Game Alternate 1 client.nlogo" ]
  hubnet-reset
  setup
end

;; initializes game
to setup
  clear-all
  setup-vars
  setup-IDs
  plotxy turns costs
  draw-map              ;; draws entire game board
  initialize-inventory         ;; places 12 cases into each roles' inventory
  set relative-speed 75  ;; Included here and not in setup-vars so that is does not run
                         ;; when NEWGAME button pushed -- allows relative-speed slider to
                         ;; remember its position when a new game is started
end

to new-game
  clear-all-plots
  clear-patches
  clear-turtles
  ask cases [die]
  ask customers [die]
  ask orders [die]
  ask backorders [die]
  setup-vars
  plotxy turns costs
  draw-map              ;; draws entire game board
  initialize-inventory  ;; places 12 cases into each roles' inventory
end

;; initializes global variables
to setup-vars
  set-default-shape cases "beer2"
  set-default-shape orders "order form"
  set-default-shape backorders "back order"
  set shipping-line1 0
  set shipping-line2 0
  set shipping-line3 0
  set shipping-line4 0
  set shipping-line5 0
  set shipping-line6 0
  set order-line1 0
  set order-line1 0
  set order-line1 0

  set costs 0
  set turns 0
  set number-of-customers 0
  set retail-inventory 0
  set distrib-inventory 0
  set manufact-inventory 0
  set retail-backorders 0
  set distrib-backorders 0
  set manufact-backorders 0
  
  set retailer-slider-value 5
  set distributor-slider-value 5
  set manufacturer-slider-value 5
end

to setup-IDs
  set retailer_ID -1        ;; IDs get set to -1 so we know they are not yet assigned
  set distributor_ID -1
  set manufacturer_ID -1
end

to draw-map        ;; draws game board's many rectangles
  cg
  ;;; retailer
  ask patches with [pxcor >= -3 and pxcor <= 3 and pycor >= 19 and pycor <= 24] [set pcolor white]
  ask patches with [pxcor >= -26 and pxcor <= -4 and pycor >= 17 and pycor <= 26] [set pcolor green]
  ask patches with [pxcor >= 4 and pxcor <= 26 and pycor >= 17 and pycor <= 26] [set pcolor red]

  ;;; distributor
  ask patches with [pxcor >= -3 and pxcor <= 3 and pycor >= -3 and pycor <= 2] [set pcolor white]
  ask patches with [pxcor >= 4 and pxcor <= 26 and pycor >= -5 and pycor <= 4] [set pcolor red]
  ask patches with [pxcor >= -26 and pxcor <= -4 and pycor >= -5 and pycor <= 4] [set pcolor green]
  
  ;;; manufacturer
  ask patches with [pxcor >= -3 and pxcor <= 3 and pycor >= -25 and pycor <= -20] [set pcolor white]
  ask patches with [pxcor >= 4 and pxcor <= 26 and pycor >= -27 and pycor <= -18] [set pcolor red]
  ask patches with [pxcor >= -26 and pxcor <= -4 and pycor >= -27 and pycor <= -18] [set pcolor green]

  ;;; order lines (1 through 3)
  ask patches with [pxcor >= -26 and pxcor <= -4 and pycor = 11 ] [set pcolor white]
  ask patches with [pxcor >= -26 and pxcor <= -4 and pycor = -11 ] [set pcolor white]
  ask patches with [pxcor >= -26 and pxcor <= -4 and pycor = -29 ] [set pcolor white]

  ;;; shipping lines (1 through 6)
  ask patches with [pxcor >= 4 and pxcor <= 26 and pycor = 13 ] [set pcolor white]
  ask patches with [pxcor >= 4 and pxcor <= 26 and pycor = 8 ] [set pcolor white]
  ask patches with [pxcor >= 4 and pxcor <= 26 and pycor = -9 ] [set pcolor white]
  ask patches with [pxcor >= 4 and pxcor <= 26 and pycor = -14 ] [set pcolor white]
  ask patches with [pxcor >= 4 and pxcor <= 26 and pycor = -28 ] [set pcolor white]
  ask patches with [pxcor >= 4 and pxcor <= 26 and pycor = -30 ] [set pcolor white]
end


to initialize-inventory    ;;; give everybody 12 cases in inventory to start game
  repeat 12 [incr-retail-inventory]
  repeat 12 [incr-distrib-inventory]
  repeat 12 [incr-manufact-inventory]
end



;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Runtime Procedures ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;

;; the Main Routine
to go
 listen-clients   ;; get commands and data from the clients

 get-demand       ;; randomize number of customers for this turn
 serve-customers  ;; deliver beer to customers from retailer's inventory
 listen-and-wait  ;; listen and pause for gameflow
 
 advance-shipments ;; move beer along the shipping lines/into inventories
 listen-and-wait
 fill-orders       ;; move orders along order lines, creating backorders if necessary
 
 calculate-costs   ;; asess penalties for backorders and cases in inventory
 set turns (turns + 1)
 hubnet-broadcast "costs" costs
 hubnet-broadcast "turns" turns
 ;; plotxy turns costs
 update-plots
 listen-and-wait
 
 ask customers [die]  ;; kill all customers every turn
 set number-of-customers 0
end


to listen-and-wait
  listen-clients       ;; gets commands from clients
  wait ( (100 - relative-speed) / 25)   ;; transform speed slider value into a delay and wait...
                                        ;; thees operations make higher slider values "faster"...
                                        ;; the default slider value of 75 creates a 1 second delay
end


to get-demand
  repeat random 12 [draw-customers]
end

to draw-customers
  cct-customers 1 [setxy ((number-of-customers mod 7) + -3) (27 - floor (number-of-customers / 7))]
  set number-of-customers (number-of-customers + 1)
end    

;; fills any customer backorders if there's beer in inventory, then
;; delivers beer to customers from retailer's inventory
to serve-customers
  repeat retail-backorders
    [ if retail-inventory > 0
      [ decr-retail-inventory  decr-retail-backorders]
    ]

  repeat number-of-customers
    [ ifelse retail-inventory > 0   ;; if the retailer has more cases than there are customers...
      [ decr-retail-inventory ]     ;; then the retailer's inventory is decreased...
      [ incr-retail-backorders ]    ;; if the retailer doesn't have enough cases, he gets backorders
    ]
end


;; advances cases from shipping lines to other shipping lines or to inventories
to advance-shipments
  repeat shipping-line1 [ incr-retail-inventory decr-shipping-line1 ]   ;; from shipping-line1 to retailer
  repeat shipping-line2 [ incr-shipping-line1 decr-shipping-line2 ]     ;; from shipping-line2 to shipping-line1
  repeat shipping-line3 [ incr-distrib-inventory decr-shipping-line3 ]  ;; from shipping-line3 to distributor
  repeat shipping-line4 [ incr-shipping-line3 decr-shipping-line4 ]     ;; from shipping-line4 to shipping-line3
  repeat shipping-line5 [ incr-manufact-inventory decr-shipping-line5 ] ;; from shipping-line5 to manufacturer
  repeat shipping-line6 [ incr-shipping-line5 decr-shipping-line6 ]     ;; from shipping-line6 to shipping-line5
end


;; Processes orders by shipping cases from inventories or creating backorders (when supplies aren't sufficient).
;; Note that a player can never have backorders and cases in inventory at the same time (because backorders may
       ;; accrue only when no cases exist in inventories, and the moment cases arrived, backordes get shipped.
to fill-orders
  repeat order-line3 [ incr-shipping-line6 decr-order-line3 ]   ;; all orders from manufacturer get shipped
 
  repeat order-line2 [ ifelse manufact-inventory > 0      ;; if manfact has enough cases to fullfill distrib's orders
                         [ incr-shipping-line4 decr-order-line2 decr-manufact-inventory ]   ;; ...he ships to distrib
                         [ incr-manufact-backorders decr-order-line2 ] ]                  ;; else, he gets backorders

  repeat order-line1 [ ifelse distrib-inventory > 0      ;; if distrib has enough cases to fullfill retailer's orders
                         [ incr-shipping-line2 decr-order-line1 decr-distrib-inventory ]   ;; ...he ships to retailer 
                         [ incr-distrib-backorders decr-order-line1 ] ]                   ;; else, he gets backorders
                         
  repeat manufact-backorders [ if manufact-inventory > 0    ;; if manufact has newly received cases, then ship out...
                             [ incr-shipping-line4 decr-manufact-backorders ] ]  ;; ...as many backorders as possible

  repeat distrib-backorders [ if distrib-inventory > 0      ;; if distrib has newly received cases, then ship out...
                            [ incr-shipping-line2 decr-distrib-backorders ] ]   ;; ...as many backorders as possible
                                
end


;;;; simple proceedures to abstract the case-ordering process
to place-retail-order
  incr-order-line1
end

to place-distrib-order
  incr-order-line2
end

to place-manufact-order
  incr-order-line3
end


;; each turn a cost of 1 is asessed for every case in inventory,
;; and a cost of 2 is asessed for every case that is backordered.
;; these costs are kept for the team as a whole and last then entire game (and hence can never decrease)
to calculate-costs
  let temp 0
  set temp (retail-backorders + distrib-backorders + manufact-backorders)
  set temp ((2 * temp) + retail-inventory + distrib-inventory + manufact-inventory)
  set costs (temp + costs)
end

to update-plots
  set-current-plot "Cost Performance"
  set-current-plot-pen "cost"
  plotxy turns costs
  set-current-plot "General Performance"
  set-current-plot-pen "demand"
  plotxy turns number-of-customers
  set-current-plot-pen "backorders"
  plotxy turns (retail-backorders + distrib-backorders + manufact-backorders)
  set-current-plot-pen "inventories"
  plotxy turns (retail-inventory + distrib-inventory + manufact-inventory)
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Code for interacting with the clients ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; determines which client sent a command, and what the command was
to listen-clients
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    ifelse (hubnet-enter-message?)
    [ create-new-student ]
    [
      ifelse hubnet-exit-message?
      [ remove-student ]
      [ execute-command hubnet-message-tag ]
    ]
  ]
end


;; Create a participant and assign him a role (retailer, distributor, or manufacturer).
;; Keep track of unassigned roles through IDs (an ID of -1 means the role is able to be assigned).
;; There is a maximum of 3 participants.  Anyone else who logs on may observe but not interact
to create-new-student
    
    ifelse retailer_ID = -1
    [ set retailer_ID hubnet-message-source]
    [
      ifelse distributor_ID = -1
      [ set distributor_ID hubnet-message-source]
      [
        ifelse manufacturer_ID = -1
        [ set manufacturer_ID hubnet-message-source]
        [hubnet-send hubnet-message-source "You are:" "only an observer."
         hubnet-send hubnet-message-source "You control:" "nothing.  Max # of participants = 3."]
      ]
    ]

    send-info-to-clients
end


;; NetLogo knows what each participant wants to 
;; do based on the tag sent by the node:
to execute-command [command]
  
  ;; placing orders
  if (command = "Place order as retailer") and (hubnet-message-source = retailer_ID)
  [ repeat  retailer-slider-value [ place-retail-order] ]
  if (command = "Place order as distributor") and (hubnet-message-source = distributor_ID)
  [ repeat  distributor-slider-value  [ place-distrib-order] ]
  if (command = "Place order as manufacturer") and (hubnet-message-source = manufacturer_ID)
  [ repeat  manufacturer-slider-value  [ place-manufact-order] ]
  
  ;; adjusting and tracking order sliders
  if command = "retailer-order-amount"
  [ set retailer-slider-value hubnet-message]
  if command = "distributor-order-amount"
  [ set distributor-slider-value hubnet-message]
  if command = "manufacturer-order-amount"
  [ set manufacturer-slider-value hubnet-message]
end


;; when a participant leaves, his role gets freed up by
;; giving his role's ID a value of -1
to remove-student
    ifelse retailer_ID = hubnet-message-source
    [ set retailer_ID -1]
    [
      ifelse distributor_ID = hubnet-message-source
      [ set distributor_ID -1]
      [
        if manufacturer_ID = hubnet-message-source
        [ set manufacturer_ID -1]
      ]
    ]
end


;; sends the appropriate monitor information back to the client
to send-info-to-clients
  ifelse hubnet-message-source = retailer_ID
  [hubnet-send retailer_ID "You are:" "the RETAILER."
   hubnet-send retailer_ID "You control:" "the Retailer's tools only."]
  [
    ifelse hubnet-message-source = distributor_ID
    [hubnet-send distributor_ID "You are:" "the DISTRIBUTOR."
     hubnet-send distributor_ID "You control:" "the Distributor's tools only."]

    [
      if hubnet-message-source = manufacturer_ID
      [hubnet-send manufacturer_ID "You are:" "the MANUFACTURER."
       hubnet-send manufacturer_ID "You control:" "the manufacturer's tools only."]
    ]
  ]
end




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Procedures for moving game pieces ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; The function of these many small proceedures can be garnered from their names.
;; At the heart of the Game are cases, backorders, and orders -- breeds of turtles
;; that move arounnd the game board.  The functions below manage the accounting and
;; movement of these turtles.  The variety of numbers found in each function may look
;; daunting, but they are for drawing purposes.  The gameboard consists of 15 different
;; rectangles on to which the turtles must jump in an organized manner.  All of this is
;; expressed using the raw coordinate system, hence the odd looking math operations.


;;;;;;;;;; Retailer ;;;;;;;;;;;;;;
to incr-retail-inventory
  cct-cases 1 [setxy ((retail-inventory mod 23) + 4) (26 - floor (retail-inventory / 23))]
  set retail-inventory (retail-inventory + 1)
end

to decr-retail-inventory
  ask cases-at (((retail-inventory - 1) mod 23) + 4) (26 - floor ((retail-inventory - 1) / 23))  [die]
  if retail-inventory > 0 [set retail-inventory (retail-inventory - 1)]
end

to incr-retail-backorders
  cct-backorders 1 [setxy ((retail-backorders mod 23) + -26) (26 - floor (retail-backorders / 23))]
  set retail-backorders (retail-backorders + 1)
end

to decr-retail-backorders
  ask backorders-at (((retail-backorders - 1) mod 23) + -26) (26 - floor ((retail-backorders - 1) / 23))  [die]
  if retail-backorders > 0 [set retail-backorders (retail-backorders - 1)]
end


;;;;;;;;;; Distributer ;;;;;;;;;;;;;;
to incr-distrib-inventory
  cct-cases 1 [setxy ((distrib-inventory mod 23) + 4) (4 - floor (distrib-inventory / 23))]
  set distrib-inventory (distrib-inventory + 1)
end

to decr-distrib-inventory
  ask cases-at (((distrib-inventory - 1) mod 23) + 4) (4 - floor ((distrib-inventory - 1) / 23))  [die]
  if distrib-inventory > 0 [set distrib-inventory (distrib-inventory - 1)]
end

to incr-distrib-backorders
  cct-backorders 1 [setxy ((distrib-backorders mod 23) + -26) (4 - floor (distrib-backorders / 23))]
  set distrib-backorders (distrib-backorders + 1)
end

to decr-distrib-backorders
  ask backorders-at (((distrib-backorders - 1) mod 23) + -26) (4 - floor ((distrib-backorders - 1) / 23))  [die]
  if distrib-backorders > 0 [set distrib-backorders (distrib-backorders - 1)]
end


;;;;;;;;;; Manufacturer ;;;;;;;;;;;;;;
to incr-manufact-inventory
  cct-cases 1 [setxy ((manufact-inventory mod 23) + 4) (-18 - floor (manufact-inventory / 23))]
  set manufact-inventory (manufact-inventory + 1)
end

to decr-manufact-inventory
  ask cases-at (((manufact-inventory - 1) mod 23) + 4) (-18 - floor ((manufact-inventory - 1) / 23))  [die]
  if manufact-inventory > 0 [set manufact-inventory (manufact-inventory - 1)]
end

to incr-manufact-backorders
  cct-backorders 1 [setxy ((manufact-backorders mod 23) + -26) (-18 - floor (manufact-backorders / 23))]
  set manufact-backorders (manufact-backorders + 1)
end

to decr-manufact-backorders
  ask backorders-at (((manufact-backorders - 1) mod 23) + -26) (-18 - floor ((manufact-backorders - 1) / 23))  [die]
  if manufact-backorders > 0 [set manufact-backorders (manufact-backorders - 1)]
end



;;;;;;;;;;;;; Moving beer along shipping lines  ;;;;;;;;;;;;;;;;;;;;
to incr-shipping-line1
  cct-cases 1 [setxy ((shipping-line1 mod 23) + 4) 13]
  set shipping-line1 (shipping-line1 + 1)
end

to decr-shipping-line1
  if shipping-line1 > 0 [
      ask one-of cases-at (((shipping-line1 - 1) mod 23) + 4) 13  [die]
      set shipping-line1 (shipping-line1 - 1)  ]
end

to incr-shipping-line2
  cct-cases 1 [setxy ((shipping-line2 mod 23) + 4) 8]
  set shipping-line2 (shipping-line2 + 1)
end

to decr-shipping-line2
  if shipping-line2 > 0 [
      ask one-of cases-at (((shipping-line2 - 1) mod 23) + 4) 8  [die]
      set shipping-line2 (shipping-line2 - 1)  ]
end

to incr-shipping-line3
  cct-cases 1 [setxy ((shipping-line3 mod 23) + 4) -9]
  set shipping-line3 (shipping-line3 + 1)
end

to decr-shipping-line3
  if shipping-line3 > 0 [
      ask one-of cases-at (((shipping-line3 - 1) mod 23) + 4) -9  [die]
      set shipping-line3 (shipping-line3 - 1)  ]
end

to incr-shipping-line4
  cct-cases 1 [setxy ((shipping-line4 mod 23) + 4) -14]
  set shipping-line4 (shipping-line4 + 1)
end

to decr-shipping-line4
  if shipping-line4 > 0 [
      ask one-of cases-at (((shipping-line4 - 1) mod 23) + 4) -14  [die]
      set shipping-line4 (shipping-line4 - 1)  ]
end

to incr-shipping-line5
  cct-cases 1 [setxy ((shipping-line5 mod 23) + 4) -28]
  set shipping-line5 (shipping-line5 + 1)
end

to decr-shipping-line5
  if shipping-line5 > 0 [
      ask one-of cases-at (((shipping-line5 - 1) mod 23) + 4) -28  [die]
      set shipping-line5 (shipping-line5 - 1)  ]
end

to incr-shipping-line6
  cct-cases 1 [setxy ((shipping-line6 mod 23) + 4) -30]
  set shipping-line6 (shipping-line6 + 1)
end

to decr-shipping-line6
  if shipping-line6 > 0 [
      ask one-of cases-at (((shipping-line6 - 1) mod 23) + 4) -30  [die]
      set shipping-line6 (shipping-line6 - 1)  ]
end


;;;;;;;;;;;;; Moving orders along processing lines  ;;;;;;;;;;;;;;;;;;;;
to incr-order-line1
  cct-orders 1 [setxy ((order-line1 mod 23) + -26) 11]
  set order-line1 (order-line1 + 1)
end

to decr-order-line1
  if order-line1 > 0 [
      ask one-of orders-at (((order-line1 - 1) mod 23) + -26) 11  [die]
      set order-line1 (order-line1 - 1)  ]
end

to incr-order-line2
  cct-orders 1 [setxy ((order-line2 mod 23) + -26) -11]
  set order-line2 (order-line2 + 1)
end

to decr-order-line2
  if order-line2 > 0 [
      ask one-of orders-at (((order-line2 - 1) mod 23) + -26) -11  [die]
      set order-line2 (order-line2 - 1)  ]
end

to incr-order-line3
  cct-orders 1 [setxy ((order-line3 mod 23) + -26) -29]
  set order-line3 (order-line3 + 1)
end

to decr-order-line3
  if order-line3 > 0 [
      ask one-of orders-at (((order-line3 - 1) mod 23) + -26) -29  [die]
      set order-line3 (order-line3 - 1)  ]
end


; *** NetLogo Model Copyright Notice ***
;
; This activity and associated models and materials was created as part of the projects:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and 
; INTEGRATED SIMULATION AND MODELING ENVIRONMENT.
; These projects gratefully acknowledge the support of the 
; National Science Foundation (REPP & ROLE programs) -- grant numbers 
; REC #9814682 and REC-0126227.
;
; Copyright 2003 by Uri Wilensky & Walter Stroup. Updated 2003. All rights reserved.
;
; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from the copyright holders.
; Contact the copyright holders for appropriate licenses for redistribution 
; for profit.
;
; To refer to this model in academic publications, please use:
; Wilensky, U. & Stroup, W. (2003).  NetLogo HubNet Beer Game Alternate 1 model.
; http://ccl.northwestern.edu/netlogo/models/HubNetBeerGameAlternate1
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.
;
; In other publications, please use:
; Copyright 2003 by Uri Wilensky and Walter Stroup.  All rights reserved.  See
; http://ccl.northwestern.edu/netlogo/models/HubNetBeerGameAlternate1
; for terms of use.
;
; *** End of NetLogo Model Copyright Notice ***
@#$#@#$#@
GRAPHICS-WINDOW
310
10
862
583
30
30
8.9
1
10
1
1
1
0

CC-WINDOW
5
597
871
692
Command Center

BUTTON
16
10
119
43
New Game
new-game
NIL
1
T
OBSERVER
T
NIL

BUTTON
127
11
230
44
Start
go
T
1
T
OBSERVER
T
NIL

MONITOR
138
126
243
175
costs
costs
0
1

MONITOR
6
126
129
175
turns
turns
0
1

TEXTBOX
538
111
628
129
        Retailer

TEXTBOX
537
307
627
325
     Distributor

TEXTBOX
538
501
628
519
    Manfacturer

PLOT
14
183
241
333
Cost Performance
turns
costs
0.0
10.0
0.0
250.0
true
false
PENS
"cost" 1.0 0 -16777216 true

SLIDER
17
71
232
104
relative-speed
relative-speed
0
100
75
5
1
NIL

TEXTBOX
28
105
118
123
slower

TEXTBOX
191
105
281
123
faster

PLOT
13
337
241
487
General Performance
turns
amount
0.0
10.0
0.0
50.0
true
false
PENS
"demand" 1.0 0 -16777216 true
"backorders" 1.0 0 -11352576 true
"inventories" 1.0 0 -65536 true

TEXTBOX
351
50
555
68
  Backorders and order processing

TEXTBOX
627
51
800
69
      Inventories and shipping

@#$#@#$#@
SPECIAL NOTE
------------
"Beer Game Alternate 1" and "Beer Game Alternate 2" are supplied with this release as examples of other possible approaches to designing and implementing a Beer Game activity.  The main Beer Game activity is the one we have developed most fully in terms of: design; reliability and robustness; documentation; and quality of code.  However, we felt that users might also be interested in seeing these other implementations.  Be cautious if intending to use the alternates with an actual group.


WHAT IS IT?
-----------
This model is a HubNet version of the Beer Game, a simple paper-based game used in economics and business management to show the difficulty in managing a supply chain.

The NetLogo model "Beer_game.nlogo" uses the NetLogo model "beer_client.nlogo" for the interfaces in the HubNet Clients that participants use.

The Beer Game was developed at MIT in the 1960s to demonstrate the surprisingly difficult troubles with lags in a supply chain. This very simple game often humbles managers and business students who perhaps initially think of the game as being trivially easy. 

There are three players in this version of the Beer Game, a retailer, a distributor, and a manufacturer. The players link to form a supply chain which delivers beer to customers through a system of orders and deliveries. The players are not in competition -- they work as team to produce the lowest possible costs. 

From where do costs arise? For each participant, there is a cost penalty of $1 for each case of beer held in inventory each turn. To reduce these costs, participants attempt to keep their inventories small. However, there is another, larger cost that "competes" with keeping a small inventory. For each backlog (i.e. a customer order which cannot immediately be filled due to low inventory), a cost penalty of $2 is assessed each turn.

There are two "catches." First, customer demand is variable. Presumably, this variability ripples through the supply chain as the retailer increases/decreases orders to the distributor who in turn increases/decreases orders to the manufacturer. Second, there is a substantial lag time for all transactions. It takes one turn for orders to be processed down the chain, and it takes two turns for deliveries to be made down the chain. The result is that orders of beer made to satisfy changes in customer demand don't execute fast enough for retailers to accurately react to customers.

NOTE: Most versions of the Beer Game only give participants local information.  Players do not see the demand, backorders, or inventories of other players.  Communication is accomplished only through the placement of orders and receipt of deliveries.  In this HubNet version, all participants are omniscient. with global information.  ALSO, the traditional Beer Game is turn based, like a board game.  This HubNet version is quicker and more fluid in order to take advantage of the medium.



HOW TO USE IT
-------------
The teacher hosts the Beer Game.  It takes three students (clients) to play the game, with a single student per role (retailer, distributor, manufacturer).  If more students log in, they will be able to watch the game but will not be able to interact.  The NEWGAME button re-initializes the game; participants who are assigned roles retain their roles even after the NEWGAME button is pressed.  The START button may pressed to start a game.  Once in a game, the start button may be toggled like a "pause" button.

BUTTONS:
--------
NEWGAME - draws the gameboard and resets all numbers and plots.  Students who remain logged in after NEWGAME is clicked retain their status.
START - runs the game.  May be toggled on and off like a "pause" button during games.

SLIDERS:
--------
RELATIVE-SPEED - controls speed of game.  Slide to the left to slow down and to the right to speed up.

MONITORS:
---------
TURNS - the number of turns through which the current game has progressed.  A turn is defined as a single cycle through the game (customers arrive, beer gets delivered, orders get processed, costs get asessed, customers leave).
COSTS - costs incurred by the enitre team for the game.  Each turn there is a $1 cost for every case in inventory and a $2 cost for every backorder.  Costs are cumulative and therefore cannot be decreased.

PLOTS:
------
COST PERFORMANCE - shows costs versus turns



CLIENT INFORMATION
------------------
After logging in, the client interface will appear for the students  If 3 students are not yet logged in, they will be assigned a role which will be described in the YOU ARE: monitor and the YOU CONTROL: monitor.  All students see all controls needed for the 3 roles, but a student may only interact with one set:

The Retailer places orders by clicking the PLACE ORDER AS RETAILER button.  The number of cases ordered is controlled by the RETAILER-ORDER-AMOUNT slider.

The Distributor places orders by clicking the PLACE ORDER AS DISTRIBUTOR button.  The number of cases ordered is controlled by the DISTRIBUTOR-ORDER-AMOUNT slider.

The Manufacturer places orders by clicking the PLACE ORDER AS MANUFACTURER button.  The number of cases ordered is controlled by the MANUFACTURER-ORDER-AMOUNT slider.

The COST monitor, TURNS monitor, and COST PERFORMANCE plot work as described above.

Students who are not interactive participants (i.e. students who logged in after all 3 roles were already claimed) still see the entire interface.  The interactive controls simply don't work for them.  They are observers who can still watch the Graphics Window, monitors, and plot. 



THINGS TO TRY
-------------
The goal of the Beer Game is simply to minimize total team costs -- but observe how difficult the game really is.  Traditionally teams do very poorly the first time playing the game, usually by having too much beer in their inventories.  Designate a certain length for games to run, say 50 turns.  Let a team play mulitple times and observe their improvement.

It might be interesting to see if teams do better on their initial attempt when:
  --told ahead of time about the "whiplash" effect of supply-chain delays
  --told ahead of time that most teams end up with inflated inventories
  --when they have watched other teams before them.

Try playing the game with different speeds.  Does having more time to make decsions improve performance?



EXTENDING THE MODEL
-------------------
As a major change, localized information could be implemented.  This would be a major change, as a different set of graphics would be need to be drawn for each player.  Perhaps a system of "invisible" turtle breeds could be used which only turn visible when in a specific player's scope.  Presumably this feature would be toggle-able by the host.

Another major change could be to make the game turn-based, like a board game.  This would involve a series of gates on the go proceedure -- each gate would be passed only when the specific player made his order.  Such a game would be slower yet easier to understand.

The customer demand is open to tweaking.  Currently it is rather basic -- a random number from 0 to 12 is simply selected each turn.  The range could be dynamic, adjustable by the host.  Perhaps the host could actually act as the customers, specifying demand each turn.



CREDITS AND REFERENCES
----------------------
This activity and associated models and materials was created as part of the projects:
PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and INTEGRATED SIMULATION AND MODELING ENVIRONMENT. These projects gratefully acknowledge the support of the National Science Foundation (REPP & ROLE programs) -- grant numbers REC #9814682 and REC-0126227.

Copyright 2003 by Uri Wilensky & Walter Stroup. All rights reserved. 

Permission to use, modify or redistribute this model is hereby granted, provided that both of the following requirements are followed: 
a) this copyright notice is included. 
b) this model will not be redistributed for profit without permission from the copyright holders. 
Contact the copyright holders for appropriate licenses for redistribution for profit. 

To refer to this model in academic publications, please use: Wilensky, U. & Stroup, W. (2003). NetLogo HubNet Beer Game Alternate 1 model. http://ccl.northwestern.edu/netlogo/models/HubNetBeerGameAlternate1. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL. 

In other publications, please use: Copyright 2003 by Uri Wilensky and Walter Stroup.  All rights reserved.  See http://ccl.northwestern.edu/netlogo/models/HubNetBeerGameAlternate1 for terms of use.
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7566196 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7566196 true true 150 285 285 225 285 75 150 135
Polygon -7566196 true true 150 135 15 75 150 15 285 75
Polygon -7566196 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7566196 true true 96 182 108
Circle -7566196 true true 110 127 80
Circle -7566196 true true 110 75 80
Line -7566196 true 150 100 80 30
Line -7566196 true 150 100 220 30

butterfly
true
0
Polygon -7566196 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7566196 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7566196 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7566196 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7566196 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7566196 true true 47 195 58
Circle -7566196 true true 195 195 58

circle
false
0
Circle -7566196 true true 30 30 240

circle 2
false
0
Circle -7566196 true true 16 16 270
Circle -16777216 true false 46 46 210

cow
false
0
Polygon -7566196 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7566196 true true 73 210 86 251 62 249 48 208
Polygon -7566196 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7566196 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7566196 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7566196 true true 60 15 75 300
Polygon -7566196 true true 90 150 270 90 90 30
Line -7566196 true 75 135 90 135
Line -7566196 true 75 45 90 45

flower
false
0
Polygon -11352576 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7566196 true true 85 132 38
Circle -7566196 true true 130 147 38
Circle -7566196 true true 192 85 38
Circle -7566196 true true 85 40 38
Circle -7566196 true true 177 40 38
Circle -7566196 true true 177 132 38
Circle -7566196 true true 70 85 38
Circle -7566196 true true 130 25 38
Circle -7566196 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -11352576 true false 189 233 219 188 249 173 279 188 234 218
Polygon -11352576 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7566196 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7566196 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7566196 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7566196 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7566196 true 150 0 150 300

pentagon
false
0
Polygon -7566196 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7566196 true true 135 90 165 300
Polygon -7566196 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7566196 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7566196 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7566196 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7566196 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7566196 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7566196 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7566196 true true 30 30 270 270

square 2
false
0
Rectangle -7566196 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7566196 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7566196 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7566196 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7566196 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7566196 true true 120 120 60

tree
false
0
Circle -7566196 true true 118 3 94
Rectangle -6524078 true false 120 195 180 300
Circle -7566196 true true 65 21 108
Circle -7566196 true true 116 41 127
Circle -7566196 true true 45 90 120
Circle -7566196 true true 104 74 152

triangle
false
0
Polygon -7566196 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7566196 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7566196 true true 4 45 195 187
Polygon -7566196 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7566196 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7566196 false true 24 174 42
Circle -7566196 false true 144 174 42
Circle -7566196 false true 234 174 42

turtle
true
0
Polygon -11352576 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -11352576 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -11352576 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -11352576 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -11352576 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7566196 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7566196 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7566196 true 150 285 150 15
Line -7566196 true 15 150 285 150
Circle -7566196 true true 120 120 60
Line -7566196 true 216 40 79 269
Line -7566196 true 40 84 269 221
Line -7566196 true 40 216 269 79
Line -7566196 true 84 40 221 269

x
false
0
Polygon -7566196 true true 270 75 225 30 30 225 75 270
Polygon -7566196 true true 30 75 75 30 270 225 225 270

back order
false
0
Rectangle -1 true false 18 16 284 284
Polygon -65536 true false 222 39 244 55 77 244 55 221
Polygon -65536 true false 46 62 73 38 249 227 222 250
Polygon -65536 true false 222 249 257 219 175 133 151 149
Polygon -65536 true false 119 152 38 68 59 52 129 143

beer
false
0
Rectangle -1 true false 17 16 285 285
Polygon -16776961 true false 127 39 121 90 88 120 88 254 188 254 188 123 157 88 154 39
Rectangle -65536 true false 90 149 185 190
Line -16777216 false 95 154 122 181
Line -16777216 false 118 155 94 180
Line -16777216 false 129 156 150 177
Line -16777216 false 149 158 131 178
Line -16777216 false 155 158 174 181
Line -16777216 false 176 158 154 180

beer2
false
0
Rectangle -1 true false 15 14 286 284
Rectangle -16776961 true false 94 98 216 276
Rectangle -16776961 true false 135 39 176 115
Rectangle -65536 true false 91 156 96 156
Rectangle -65536 true false 95 157 215 240
Rectangle -16776961 true false 95 154 215 253
Rectangle -16776961 true false 71 85 233 278
Rectangle -16776961 true false 115 29 194 103
Rectangle -1 true false 56 261 244 280
Rectangle -1 true false 62 252 245 268
Rectangle -16776961 true false 233 86 242 251
Rectangle -1 true false 95 22 218 84
Rectangle -16776961 true false 132 37 178 109

beer3
false
0
Rectangle -1 true false 33 17 273 284
Rectangle -16776961 true false 43 27 263 274
Rectangle -65536 true false 45 106 261 196

order form
false
0
Rectangle -1 true false 18 15 282 284
Polygon -16777216 true false 43 185 82 211 100 231 108 250 257 46 235 33 122 193 113 214 64 166 51 169
Polygon -16777216 true false 52 168 64 152 122 198 110 219

order form2
false
0
Rectangle -1 true false 15 14 285 284
Rectangle -16777216 false false 38 32 263 256
Rectangle -16777216 false false 40 34 262 255
Rectangle -16777216 false false 39 33 261 254
Rectangle -16777216 false false 36 31 264 256
Rectangle -16777216 false false 38 31 265 257
Rectangle -16777216 false false 36 31 260 253
Rectangle -16777216 false false 37 31 258 252
Rectangle -16777216 false false 34 31 259 251
Rectangle -16777216 true false 25 28 41 257
Rectangle -16777216 true false 30 236 265 256
Rectangle -16777216 true false 27 28 264 46
Rectangle -16777216 true false 247 28 265 255
Polygon -16777216 true false 54 82 84 69 113 83 146 71 173 84 207 70 230 82 206 75 180 86 171 88 148 77 113 89 87 74
Polygon -16777216 true false 53 128 98 118 148 132 186 119 232 128 233 138 188 127 149 140 98 125 59 136
Rectangle -16777216 true false 49 158 116 169
Rectangle -16777216 true false 166 160 231 168
Rectangle -16777216 true false 52 180 115 190
Rectangle -16777216 true false 166 178 231 186
Rectangle -16777216 true false 170 196 229 202
Rectangle -16777216 true false 52 199 110 207
Line -16777216 false 53 226 54 225
Polygon -16777216 true false 56 227 101 217 133 221 174 217 197 225 231 217 212 228 198 230 172 222 141 227 99 223 69 231
Polygon -16777216 true false 61 100 213 101 211 106
Polygon -16777216 true false 52 101 130 101 130 109
Polygon -16777216 true false 209 76 229 85 233 81
Polygon -16777216 true false 86 76 64 88 55 81

order form3
false
0
Rectangle -1 true false 14 14 284 284
Rectangle -16777216 true false 37 38 260 54
Rectangle -16777216 true false 42 102 257 120
Rectangle -16777216 true false 46 167 258 183
Rectangle -16777216 true false 51 226 259 244
Rectangle -16777216 true false 44 163 262 190
Rectangle -16777216 true false 38 223 262 249
Rectangle -1 true false 240 28 267 260
Rectangle -16777216 true false 40 100 239 122
Rectangle -16777216 true false 35 36 240 56
Rectangle -1 true false 23 27 255 256
Rectangle -16777216 true false 51 50 242 77
Rectangle -16777216 true false 58 145 240 171
Rectangle -16777216 true false 54 229 239 250
Rectangle -1 true false 226 32 249 258
Rectangle -16777216 true false 55 219 226 247
Rectangle -1 true false 50 132 245 187
Rectangle -16777216 true false 52 133 226 161
Rectangle -1 true false 222 42 232 252
Rectangle -1 true false 26 28 265 267
Rectangle -16777216 true false 48 66 247 115
Rectangle -16777216 true false 57 179 250 225
Rectangle -1 true false 223 41 258 237
Rectangle -1 true false 38 46 76 241
@#$#@#$#@
NetLogo 2.1beta2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
